/*
 * Decompiled with CFR 0.152.
 */
package com.file.netty.utils;

import com.file.netty.protocol.CompressedBatchManifestMessage;
import com.file.netty.utils.FileSizeCategory;
import com.file.netty.utils.FileTransferUtils;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompressedBatchBuilder {
    private static final Logger logger = LoggerFactory.getLogger(CompressedBatchBuilder.class);
    public static final long MAX_BATCH_SIZE = 0x6400000L;
    public static final long MIN_BATCH_SIZE = 0x100000L;
    private static final Map<String, Double> COMPRESSION_RATIOS = new HashMap<String, Double>();

    public static List<CompressedBatch> buildCompressedBatches(List<File> files, String baseDir, AtomicInteger batchIdCounter) {
        List<File> compressionFiles;
        ArrayList<CompressedBatch> batches = new ArrayList<CompressedBatch>();
        List sortedFiles = files.stream().filter(File::isFile).sorted(Comparator.comparingLong(File::length)).collect(Collectors.toList());
        logger.info("\u5f00\u59cb\u6784\u5efa\u538b\u7f29\u6279\u6b21: \u603b\u5171 {} \u4e2a\u6587\u4ef6\uff0c\u6309\u5927\u5c0f\u6392\u5e8f", (Object)sortedFiles.size());
        ArrayList<File> smallFiles = new ArrayList<File>();
        ArrayList<File> largeFiles = new ArrayList<File>();
        for (File file : sortedFiles) {
            if (file.length() > 0x6400000L) {
                largeFiles.add(file);
                continue;
            }
            smallFiles.add(file);
        }
        logger.info("\u6587\u4ef6\u5206\u7c7b: \u5c0f\u6587\u4ef6(<100MB) {}\u4e2a, \u5927\u6587\u4ef6(>100MB) {}\u4e2a", (Object)smallFiles.size(), (Object)largeFiles.size());
        if (!smallFiles.isEmpty() && !(compressionFiles = CompressedBatchBuilder.selectFilesForCompression(smallFiles)).isEmpty()) {
            int currentBatchId = batchIdCounter.getAndIncrement();
            long totalSize = compressionFiles.stream().mapToLong(File::length).sum();
            String batchName = String.format("\ud83d\udce6\u5c0f\u6587\u4ef6\u6279\u6b21%d", currentBatchId);
            CompressedBatch batch = new CompressedBatch(currentBatchId, batchName, compressionFiles, FileSizeCategory.SMALL, baseDir);
            batches.add(batch);
            logger.info("\u521b\u5efa\u538b\u7f29\u6279\u6b21: {} - {} \u4e2a\u6587\u4ef6, \u603b\u5927\u5c0f: {}", batchName, compressionFiles.size(), FileTransferUtils.formatFileSize(totalSize));
        }
        if (!largeFiles.isEmpty()) {
            logger.info("\u5927\u6587\u4ef6\u5c06\u4f7f\u7528\u666e\u901a\u4f20\u8f93: {} \u4e2a\u6587\u4ef6", (Object)largeFiles.size());
            for (File file : largeFiles) {
                logger.debug("\u5927\u6587\u4ef6: {} ({})", (Object)file.getName(), (Object)FileTransferUtils.formatFileSize(file.length()));
            }
        }
        logger.info("\u603b\u5171\u6784\u5efa\u4e86 {} \u4e2a\u538b\u7f29\u6279\u6b21", (Object)batches.size());
        return batches;
    }

    private static List<File> selectFilesForCompression(List<File> sortedFiles) {
        int MIN_COUNT = 5;
        int MAX_COUNT = 5000;
        long TARGET_SIZE = 0x6400000L;
        if (sortedFiles.size() < 5) {
            return new ArrayList<File>();
        }
        int maxSelectableCount = Math.min(sortedFiles.size(), 5000);
        long bestSize = 0L;
        int bestCount = 0;
        long totalSize = 0L;
        for (int i = 0; i < maxSelectableCount; ++i) {
            totalSize += sortedFiles.get(i).length();
            if (i + 1 < 5) continue;
            if (bestCount == 0 || Math.abs(totalSize - 0x6400000L) < Math.abs(bestSize - 0x6400000L)) {
                bestSize = totalSize;
                bestCount = i + 1;
            }
            if ((double)totalSize > 1.572864E8) break;
        }
        if (bestCount < 5) {
            return new ArrayList<File>();
        }
        ArrayList<File> result = new ArrayList<File>(sortedFiles.subList(0, bestCount));
        logger.info("\u667a\u80fd\u9009\u62e9\u538b\u7f29\u6587\u4ef6: {} \u4e2a\u6587\u4ef6, \u5408\u8ba1 {} (\u63a5\u8fd1100MB\u76ee\u6807)", (Object)result.size(), (Object)FileTransferUtils.formatFileSize(bestSize));
        return result;
    }

    private static Map<FileSizeCategory, List<File>> groupFilesByCategory(List<File> files) {
        EnumMap<FileSizeCategory, List<File>> groups = new EnumMap<FileSizeCategory, List<File>>(FileSizeCategory.class);
        for (File file : files) {
            if (!file.isFile()) continue;
            FileSizeCategory category = FileSizeCategory.categorizeFile(file.length());
            groups.computeIfAbsent(category, k -> new ArrayList()).add(file);
        }
        return groups;
    }

    private static Map<String, List<File>> groupFilesByCompressionType(List<File> files) {
        HashMap<String, List<File>> groups = new HashMap<String, List<File>>();
        for (File file : files) {
            String compressionType = CompressedBatchBuilder.getCompressionType(file);
            groups.computeIfAbsent(compressionType, k -> new ArrayList()).add(file);
        }
        return groups;
    }

    private static List<List<File>> createSizeLimitedBatches(List<File> files) {
        ArrayList<List<File>> batches = new ArrayList<List<File>>();
        files.sort(Comparator.comparing(File::length));
        ArrayList<File> currentBatch = new ArrayList<File>();
        long currentBatchSize = 0L;
        for (File file : files) {
            long fileSize = file.length();
            if (fileSize > 0x6400000L) {
                if (!currentBatch.isEmpty()) {
                    batches.add(new ArrayList(currentBatch));
                    currentBatch.clear();
                    currentBatchSize = 0L;
                }
                batches.add(Arrays.asList(file));
                continue;
            }
            if (currentBatchSize + fileSize > 0x6400000L && !currentBatch.isEmpty()) {
                batches.add(new ArrayList(currentBatch));
                currentBatch.clear();
                currentBatchSize = 0L;
            }
            currentBatch.add(file);
            currentBatchSize += fileSize;
        }
        if (!currentBatch.isEmpty()) {
            batches.add(currentBatch);
        }
        return batches;
    }

    private static String getCompressionType(File file) {
        String name = file.getName().toLowerCase();
        int dotIndex = name.lastIndexOf(46);
        if (dotIndex == -1) {
            return "\u65e0\u6269\u5c55\u540d";
        }
        String extension = name.substring(dotIndex + 1);
        if (CompressedBatchBuilder.isTextExtension(extension)) {
            return "\u6587\u672c";
        }
        if (CompressedBatchBuilder.isCompressedExtension(extension)) {
            return "\u5df2\u538b\u7f29";
        }
        if (CompressedBatchBuilder.isMediaExtension(extension)) {
            return "\u5a92\u4f53";
        }
        return "\u4e8c\u8fdb\u5236";
    }

    private static double estimateCompressionRatio(List<File> files) {
        long totalSize = 0L;
        double weightedRatio = 0.0;
        for (File file : files) {
            long fileSize = file.length();
            totalSize += fileSize;
            double ratio = CompressedBatchBuilder.getFileCompressionRatio(file);
            weightedRatio += (double)fileSize * ratio;
        }
        return totalSize > 0L ? weightedRatio / (double)totalSize : 0.5;
    }

    private static double getFileCompressionRatio(File file) {
        String name = file.getName().toLowerCase();
        int dotIndex = name.lastIndexOf(46);
        if (dotIndex != -1) {
            String extension = name.substring(dotIndex + 1);
            return COMPRESSION_RATIOS.getOrDefault(extension, 0.6);
        }
        return 0.6;
    }

    private static boolean isTextFile(File file) {
        String name = file.getName().toLowerCase();
        int dotIndex = name.lastIndexOf(46);
        if (dotIndex != -1) {
            String extension = name.substring(dotIndex + 1);
            return CompressedBatchBuilder.isTextExtension(extension);
        }
        return false;
    }

    private static boolean isTextExtension(String ext) {
        return Arrays.asList("txt", "java", "xml", "json", "css", "js", "html", "md", "log", "csv", "properties", "yml", "yaml", "conf").contains(ext);
    }

    private static boolean isCompressedExtension(String ext) {
        return Arrays.asList("zip", "jar", "gz", "7z", "rar", "tar").contains(ext);
    }

    private static boolean isMediaExtension(String ext) {
        return Arrays.asList("jpg", "png", "gif", "mp3", "mp4", "avi", "pdf").contains(ext);
    }

    private static String getRelativePath(String baseDir, String fullPath) {
        File base = new File(baseDir);
        File full = new File(fullPath);
        try {
            return base.toPath().relativize(full.toPath()).toString().replace('\\', '/');
        }
        catch (Exception e) {
            return full.getName();
        }
    }

    private static List<List<File>> createBatchesBySize(List<File> files, long maxBatchSize) {
        ArrayList<List<File>> batches = new ArrayList<List<File>>();
        ArrayList<File> currentBatch = new ArrayList<File>();
        long currentBatchSize = 0L;
        for (File file : files) {
            long fileSize = file.length();
            if (currentBatchSize + fileSize > maxBatchSize && !currentBatch.isEmpty()) {
                batches.add(new ArrayList(currentBatch));
                logger.debug("\u5b8c\u6210\u6279\u6b21: {} \u4e2a\u6587\u4ef6, \u603b\u5927\u5c0f: {}", (Object)currentBatch.size(), (Object)FileTransferUtils.formatFileSize(currentBatchSize));
                currentBatch.clear();
                currentBatchSize = 0L;
            }
            currentBatch.add(file);
            currentBatchSize += fileSize;
        }
        if (!currentBatch.isEmpty()) {
            batches.add(currentBatch);
            logger.debug("\u5b8c\u6210\u6700\u540e\u6279\u6b21: {} \u4e2a\u6587\u4ef6, \u603b\u5927\u5c0f: {}", (Object)currentBatch.size(), (Object)FileTransferUtils.formatFileSize(currentBatchSize));
        }
        return batches;
    }

    static {
        COMPRESSION_RATIOS.put("txt", 0.3);
        COMPRESSION_RATIOS.put("java", 0.3);
        COMPRESSION_RATIOS.put("xml", 0.2);
        COMPRESSION_RATIOS.put("json", 0.3);
        COMPRESSION_RATIOS.put("css", 0.3);
        COMPRESSION_RATIOS.put("js", 0.4);
        COMPRESSION_RATIOS.put("html", 0.3);
        COMPRESSION_RATIOS.put("md", 0.4);
        COMPRESSION_RATIOS.put("log", 0.2);
        COMPRESSION_RATIOS.put("csv", 0.3);
        COMPRESSION_RATIOS.put("properties", 0.4);
        COMPRESSION_RATIOS.put("yml", 0.4);
        COMPRESSION_RATIOS.put("yaml", 0.4);
        COMPRESSION_RATIOS.put("conf", 0.4);
        COMPRESSION_RATIOS.put("zip", 0.95);
        COMPRESSION_RATIOS.put("jar", 0.95);
        COMPRESSION_RATIOS.put("gz", 0.95);
        COMPRESSION_RATIOS.put("7z", 0.95);
        COMPRESSION_RATIOS.put("jpg", 0.9);
        COMPRESSION_RATIOS.put("png", 0.9);
        COMPRESSION_RATIOS.put("gif", 0.9);
        COMPRESSION_RATIOS.put("mp3", 0.95);
        COMPRESSION_RATIOS.put("mp4", 0.95);
        COMPRESSION_RATIOS.put("class", 0.7);
        COMPRESSION_RATIOS.put("exe", 0.8);
        COMPRESSION_RATIOS.put("dll", 0.8);
        COMPRESSION_RATIOS.put("so", 0.8);
    }

    public static class CompressedBatch {
        private final int batchId;
        private final String batchName;
        private final List<File> files;
        private final FileSizeCategory category;
        private final long totalSize;
        private final double estimatedCompressionRatio;
        private final long estimatedCompressedSize;

        public CompressedBatch(int batchId, String batchName, List<File> files, FileSizeCategory category, String baseDir) {
            this.batchId = batchId;
            this.batchName = batchName;
            this.files = new ArrayList<File>(files);
            this.category = category;
            this.totalSize = files.stream().mapToLong(File::length).sum();
            this.estimatedCompressionRatio = CompressedBatchBuilder.estimateCompressionRatio(files);
            this.estimatedCompressedSize = (long)((double)this.totalSize * this.estimatedCompressionRatio);
        }

        public CompressedBatchManifestMessage createManifest(String sessionId, String baseDir) {
            ArrayList<CompressedBatchManifestMessage.BatchFileManifest> manifests = new ArrayList<CompressedBatchManifestMessage.BatchFileManifest>();
            for (File file : this.files) {
                String md5 = null;
                Exception lastException = null;
                for (int retry = 0; retry < 3; ++retry) {
                    try {
                        md5 = FileTransferUtils.calculateFileMD5(file);
                        break;
                    }
                    catch (Exception e) {
                        lastException = e;
                        if (retry >= 2) continue;
                        try {
                            Thread.sleep(100 * (retry + 1));
                            continue;
                        }
                        catch (InterruptedException ie) {
                            Thread.currentThread().interrupt();
                            break;
                        }
                    }
                }
                if (md5 != null) {
                    try {
                        String relativePath = CompressedBatchBuilder.getRelativePath(baseDir, file.getAbsolutePath());
                        boolean isTextFile = CompressedBatchBuilder.isTextFile(file);
                        CompressedBatchManifestMessage.BatchFileManifest manifest = new CompressedBatchManifestMessage.BatchFileManifest(relativePath, file.length(), md5, file.lastModified(), isTextFile);
                        manifests.add(manifest);
                    }
                    catch (Exception e) {
                        logger.error("\u751f\u6210\u6587\u4ef6\u6e05\u5355\u6761\u76ee\u5931\u8d25: {}", (Object)file.getAbsolutePath(), (Object)e);
                    }
                    continue;
                }
                logger.error("\u6587\u4ef6MD5\u8ba1\u7b973\u6b21\u91cd\u8bd5\u5747\u5931\u8d25\uff0c\u8df3\u8fc7\u6587\u4ef6: {} - \u6700\u540e\u9519\u8bef: {}", (Object)file.getAbsolutePath(), (Object)(lastException != null ? lastException.getMessage() : "\u672a\u77e5\u9519\u8bef"));
            }
            return new CompressedBatchManifestMessage(sessionId, this.batchId, this.batchName, manifests);
        }

        public int getBatchId() {
            return this.batchId;
        }

        public String getBatchName() {
            return this.batchName;
        }

        public List<File> getFiles() {
            return this.files;
        }

        public FileSizeCategory getCategory() {
            return this.category;
        }

        public long getTotalSize() {
            return this.totalSize;
        }

        public double getEstimatedCompressionRatio() {
            return this.estimatedCompressionRatio;
        }

        public long getEstimatedCompressedSize() {
            return this.estimatedCompressedSize;
        }

        public String getSizeInfo() {
            return String.format("%s \u2192 %s (\u9884\u4f30%.1f%%\u8282\u7701)", FileTransferUtils.formatBytes(this.totalSize), FileTransferUtils.formatBytes(this.estimatedCompressedSize), (1.0 - this.estimatedCompressionRatio) * 100.0);
        }
    }
}

