/*
 * Decompiled with CFR 0.152.
 */
package com.file.netty.utils;

import com.file.netty.utils.FileTransferUtils;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ParallelHashCalculator {
    private static final Logger logger = LoggerFactory.getLogger(ParallelHashCalculator.class);
    private static final int CPU_CORES = Runtime.getRuntime().availableProcessors();
    private static final int DEFAULT_THREAD_COUNT = Math.max(4, CPU_CORES * 2);
    private static final int MAX_THREAD_COUNT = Math.max(16, CPU_CORES * 3);
    private static final long LARGE_FILE_THRESHOLD = 0x100000L;

    public static HashResult calculateHashesParallel(List<File> files, ProgressCallback callback) {
        return ParallelHashCalculator.calculateHashesParallel(files, DEFAULT_THREAD_COUNT, callback);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static HashResult calculateHashesParallel(List<File> files, int threadCount, ProgressCallback callback) {
        if (files == null || files.isEmpty()) {
            return new HashResult(new ConcurrentHashMap<File, String>(), new ConcurrentHashMap<File, Exception>(), 0L, 0.0);
        }
        threadCount = Math.min(threadCount, MAX_THREAD_COUNT);
        threadCount = Math.min(threadCount, files.size());
        long startTime = System.currentTimeMillis();
        ArrayList<File> sortedFiles = new ArrayList<File>(files);
        sortedFiles.sort((a, b) -> {
            if (a.length() <= 1024L && b.length() > 1024L) {
                return -1;
            }
            if (b.length() <= 1024L && a.length() > 1024L) {
                return 1;
            }
            return Long.compare(a.length(), b.length());
        });
        ConcurrentHashMap<File, String> hashMap = new ConcurrentHashMap<File, String>();
        ConcurrentHashMap<File, Exception> errorMap = new ConcurrentHashMap<File, Exception>();
        AtomicInteger completedCount = new AtomicInteger(0);
        AtomicLong totalHashTime = new AtomicLong(0L);
        ThreadPoolExecutor executor = new ThreadPoolExecutor(threadCount, threadCount, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(), r -> {
            Thread t = new Thread(r, "HashCalculator-" + Thread.currentThread().getId());
            t.setDaemon(true);
            return t;
        });
        try {
            ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>();
            for (File file : sortedFiles) {
                Future<Void> future = executor.submit(() -> {
                    try {
                        long fileStartTime = System.currentTimeMillis();
                        if (callback != null) {
                            callback.onProgress(completedCount.get(), files.size(), file.getName(), file.length(), System.currentTimeMillis() - startTime);
                        }
                        String md5 = null;
                        Exception lastException = null;
                        for (int retry = 0; retry < 3; ++retry) {
                            try {
                                md5 = FileTransferUtils.calculateFileMD5(file);
                                break;
                            }
                            catch (Exception e) {
                                lastException = e;
                                if (retry >= 2) continue;
                                try {
                                    Thread.sleep(100 * (retry + 1));
                                    continue;
                                }
                                catch (InterruptedException ie) {
                                    Thread.currentThread().interrupt();
                                    break;
                                }
                            }
                        }
                        if (md5 == null) {
                            throw lastException != null ? lastException : new RuntimeException("MD5\u8ba1\u7b97\u5931\u8d25");
                        }
                        hashMap.put(file, md5);
                        long fileTime = System.currentTimeMillis() - fileStartTime;
                        totalHashTime.addAndGet(fileTime);
                        if (callback != null) {
                            callback.onFileCompleted(file, md5, fileTime);
                        }
                        int completed = completedCount.incrementAndGet();
                        if (callback != null) {
                            callback.onProgress(completed, files.size(), file.getName(), file.length(), System.currentTimeMillis() - startTime);
                        }
                    }
                    catch (Exception e) {
                        errorMap.put(file, e);
                        if (callback != null) {
                            callback.onFileError(file, e);
                        }
                        completedCount.incrementAndGet();
                        logger.warn("\u8ba1\u7b97\u6587\u4ef6MD5\u5931\u8d25: {} - {}", (Object)file.getAbsolutePath(), (Object)e.getMessage());
                    }
                    return null;
                });
                futures.add(future);
            }
            for (Future future : futures) {
                try {
                    future.get(5L, TimeUnit.MINUTES);
                }
                catch (TimeoutException e) {
                    logger.error("\u6587\u4ef6MD5\u8ba1\u7b97\u8d85\u65f6\uff0c\u53d6\u6d88\u5269\u4f59\u4efb\u52a1");
                    future.cancel(true);
                }
                catch (ExecutionException e) {
                    logger.error("\u6587\u4ef6MD5\u8ba1\u7b97\u5f02\u5e38", e.getCause());
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    logger.error("\u6587\u4ef6MD5\u8ba1\u7b97\u88ab\u4e2d\u65ad");
                    future.cancel(true);
                }
            }
        }
        finally {
            executor.shutdown();
            try {
                if (!executor.awaitTermination(30L, TimeUnit.SECONDS)) {
                    executor.shutdownNow();
                }
            }
            catch (InterruptedException e) {
                executor.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
        long totalTime = System.currentTimeMillis() - startTime;
        double d = totalTime > 0L ? (double)files.size() * 1000.0 / (double)totalTime : 0.0;
        logger.info("\ud83d\udd10 \u5e76\u884cMD5\u8ba1\u7b97\u5b8c\u6210: {}\u4e2a\u6587\u4ef6, \u6210\u529f:{}, \u5931\u8d25:{}, \u8017\u65f6:{}ms, \u901f\u5ea6:{:.1f}\u6587\u4ef6/\u79d2, \u7ebf\u7a0b\u6570:{}", files.size(), hashMap.size(), errorMap.size(), totalTime, d, threadCount);
        return new HashResult(hashMap, errorMap, totalTime, d);
    }

    public static interface ProgressCallback {
        public void onProgress(int var1, int var2, String var3, long var4, long var6);

        public void onFileCompleted(File var1, String var2, long var3);

        public void onFileError(File var1, Exception var2);
    }

    public static class HashResult {
        private final Map<File, String> hashMap;
        private final Map<File, Exception> errorMap;
        private final long totalTimeMs;
        private final double avgSpeed;

        public HashResult(Map<File, String> hashMap, Map<File, Exception> errorMap, long totalTimeMs, double avgSpeed) {
            this.hashMap = hashMap;
            this.errorMap = errorMap;
            this.totalTimeMs = totalTimeMs;
            this.avgSpeed = avgSpeed;
        }

        public Map<File, String> getHashMap() {
            return this.hashMap;
        }

        public Map<File, Exception> getErrorMap() {
            return this.errorMap;
        }

        public long getTotalTimeMs() {
            return this.totalTimeMs;
        }

        public double getAvgSpeed() {
            return this.avgSpeed;
        }

        public boolean hasErrors() {
            return !this.errorMap.isEmpty();
        }

        public int getSuccessCount() {
            return this.hashMap.size();
        }

        public int getErrorCount() {
            return this.errorMap.size();
        }
    }

    public static class ConsoleProgressCallback
    implements ProgressCallback {
        private final String prefix;
        private long lastProgressTime = 0L;
        private static final long PROGRESS_INTERVAL = 2000L;

        public ConsoleProgressCallback(String prefix) {
            this.prefix = prefix;
        }

        @Override
        public void onProgress(int completed, int total, String currentFile, long fileSize, long elapsedMs) {
            long now = System.currentTimeMillis();
            if (now - this.lastProgressTime < 2000L && completed < total) {
                return;
            }
            this.lastProgressTime = now;
            double percentage = (double)completed * 100.0 / (double)total;
            double speed = elapsedMs > 0L ? (double)completed * 1000.0 / (double)elapsedMs : 0.0;
            String etaString = "";
            if (speed > 0.0 && completed > 0) {
                int remaining = total - completed;
                double etaSeconds = (double)remaining / speed;
                etaString = etaSeconds < 60.0 ? String.format("| \u9884\u8ba1\u5269\u4f59:%.0f\u79d2", etaSeconds) : String.format("| \u9884\u8ba1\u5269\u4f59:%.1f\u5206\u949f", etaSeconds / 60.0);
            }
            System.out.printf("\ud83d\udd10 %s: MD5\u5e76\u884c\u8ba1\u7b97 [%d/%d] (%.1f%%) | \u901f\u5ea6:%.1f\u6587\u4ef6/\u79d2 %s\n", this.prefix, completed, total, percentage, speed, etaString);
            System.out.flush();
        }

        @Override
        public void onFileCompleted(File file, String md5, long timeMs) {
            if (timeMs > 1000L) {
                System.out.printf("\ud83d\udd10 %s: MD5\u5b8c\u6210 %s (\u8017\u65f6:%.1fs)\n", this.prefix, file.getName(), (double)timeMs / 1000.0);
                System.out.flush();
            }
        }

        @Override
        public void onFileError(File file, Exception error) {
            System.out.printf("\u274c %s: MD5\u8ba1\u7b97\u5931\u8d25 %s - %s\n", this.prefix, file.getName(), error.getMessage());
            System.out.flush();
        }
    }
}

